<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\BlogPost;
use App\Models\TeamMember;
use BotMan\BotMan\BotMan;
use BotMan\BotMan\Messages\Incoming\Answer;
use Intervention\Image\Facades\Image;


class LandingController extends Controller
{
    public function showLanding()
    {
        // $blogs = BlogPost::latest()->take(3)->get();
        // $staff = TeamMember::latest();
        // return view('web.index', compact('blogs'));
        $blogs = BlogPost::latest()->take(3)->get();
        $staff = TeamMember::latest()->get(); // Added ->get() to execute the query
        return view('web.index', compact('blogs', 'staff')); 
    }

    // BotMan chatbot functionality
    public function handle()
    {
        $botman = app('botman');
        // Listen for any message
        $botman->hears('{message}', function($botman, $message) {
            $message = strtolower($message);
            if ($message == 'hi') {
                $this->askName($botman);
            }
            else {
                $botman->reply("Start a conversation by saying hi.");
            }
        });
        $botman->listen();
    }

    /**
     * Ask the user for their name when they say 'hi'.
     */
    public function askName($botman)
    {
        // For fewer questions, you can use the inline conversation approach as shown below. Alternatively, use a dedicated conversation class for multi-step conversations
        $botman->ask('Hello! What is your name?', function(Answer $answer, $conversation) {
            // Capture the user's answer
            $name = $answer->getText();
            // Respond with a personalized message
            $this->say('Nice to meet you, ' . $name);
            //Continue inline conversation.
            $conversation->ask('Can you advise about your email address.', function(Answer $answer, $conversation){
                $email = $answer->getText();
                $this->say('Email : '.$email);
            });
        });
    }

    public function getNews(){
        return view('website.blog');
    }
    
    public function getProducts(){
        $products = Product::all();
        return view('web.products', compact('products'));
    }
    
    public function getTechnology(){
        return view('web.tech');
    }
    
    public function getAbout(){
        return view('web.about');
    }
    
    public function getContact(){
        return view('web.tech');
    }
    
    public function getAwards(){
        return view('web.awards');
    }
    public function getFAQs(){
        return view('web.faqs');
    }
    
    public function getProfile(){
        return view('web.profile');
    }
    
    public function getImpact(){
        return view('web.impact');
    }
    
    public function getGallery(){
        return view('web.gallery');
    }
    
    public function getProductDetails($id){
        $product = Product::findOrFail($id);
        return view('web.product-details', compact('product'));
    }

    public function listProducts(){
        $units = [
            [
                'title' => '500L / Day Air-to-Water Unit',
                'description' => 'Our units are designed for Sub Saharan conditions and perform beyond their capacities in favourable climate (high relative humidity, high annual temperatures).',
                'applications' => [
                    'Specialized farming with drip irrigation',
                    'Off-grid water for niche industry (paired with solar PV power)',
                    'Drinking water for communities far from water grid or clean drinking water source (requires filtration and sterile storage tank)',
                ],
                'image' => 'assets/web/img/products/img02.jpeg',
                'specifications' => [
                    ['Rated Water production capacity in 24 hrs.', '500 litres'],
                    ['Working conditions for rated capacity', '25 - 32°C & 70 -75% RH'],
                    ['Actual Size (L x W x H) in cm', '150 x 73 x 136'],
                    ['Power input kW', '7.5'],
                ],
            ],
            [
                'title' => '120L / Day Air-to-Water Unit',
                'description' => 'Our units are designed for Sub Saharan conditions and perform beyond their capacities in favorable climate.',
                'applications' => [
                    'Drinking water for household or office consumption.',
                    'Small scale drip irrigation or domestic water use.',
                ],
                'image' => 'assets/web/img/products/img03.jpeg',
                'specifications' => [
                    ['Rated water production capacity in 24 hrs.', '120 litres'],
                    ['Actual size (LxWxH) (cm)', '75 x 69 x 111'],
                    ['Power consumption with chiller (kW per hr)', '2.5'],
                    ['Internal storage tank capacity (litres)', '50'],
                ],
            ],
            [
                'title' => '25L / Day Air-to-Water Unit',
                'description' => 'Our units are designed for Sub Saharan conditions and perform beyond their capacities in favorable climate.',
                'applications' => ['Drinking water for household or office consumption.'],
                'image' => 'assets/web/img/products/img1.jpeg',
                'specifications' => [
                    ['Rated Water production capacity in 24 hrs.', '25 litres'],
                    ['Working conditions for rated capacity', '25 - 32°C & 80% RH'],
                    ['Actual Size (L x W x H) in cm', '35.2 x 44.2 x 108.3'],
                    ['Power consumption without chiller in kW per hr', '0.4'],
                ],
            ],
        ];

        return view('web.products');
    }

    public function getAddTeamMember()
    {
        return view('majik.addteammember');
    }

    /**
     * Store a new team member
     */
    public function store(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|string|max:255',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $imagePath = null;

        if ($request->hasFile('image')) {
            $image = $request->file('image');

            // Resize and encode the image
            $resizedImage = Image::make($image)->resize(250, 250)->encode('jpg');

            $targetDirectory = public_path('assets/web-new/assets/images/team/');
            $fileName = uniqid() . '.jpg';
            $imagePath = 'assets/web-new/assets/images/team/' . $fileName;

            // Create directory if it doesn't exist
            if (!file_exists($targetDirectory)) {
                mkdir($targetDirectory, 0777, true);
            }
            
            // Save the image
            $resizedImage->save($targetDirectory . $fileName);
        }

        // Create the team member
        TeamMember::create([
            'name' => $request->name,
            'role' => $request->role,
            'image_path' => $imagePath, 
        ]);

        return redirect()->route('teammemberlist')->with('success', 'Team member added successfully.');
    }

    /**
     * Get all team members
     */
    public function getTeamMembers()
    {
        $teamMembers = TeamMember::all();
        return response()->json($teamMembers);
        
    }

   
    public function getTeamMember($id)
    {
        $teamMember = TeamMember::findOrFail($id);
        return response()->json($teamMember);
    }

    /**
     * Update team member data
     */
    public function updateTeamMember(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $teamMember = TeamMember::findOrFail($id);
        $data = $request->only(['name', 'role']);

        // Handle image update if provided
        if ($request->hasFile('image')) {
            $image = $request->file('image');
            $resizedImage = Image::make($image)->resize(250, 250)->encode('jpg');
            
            $targetDirectory = public_path('assets/web-new/assets/images/team/');
            $fileName = uniqid() . '.jpg';
            $imagePath = 'assets/web-new/assets/images/team/' . $fileName;

            if (!file_exists($targetDirectory)) {
                mkdir($targetDirectory, 0777, true);
            }
            $resizedImage->save($targetDirectory . $fileName);
            
            // Delete old image if it exists
            if ($teamMember->image_path && file_exists(public_path($teamMember->image_path))) {
                unlink(public_path($teamMember->image_path));
            }
            
            $data['image_path'] = $imagePath;
        }

        $teamMember->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Team member updated successfully'
        ]);
    }

    /**
     * Delete team member
     */
    public function deleteTeamMember($id)
    {
        $teamMember = TeamMember::findOrFail($id);
        
        // Delete associated image
        if ($teamMember->image_path && file_exists(public_path($teamMember->image_path))) {
            unlink(public_path($teamMember->image_path));
        }
        
        $teamMember->delete();

        return response()->json([
            'success' => true,
            'message' => 'Team member deleted successfully'
        ]);
    }
}